<?php

namespace App\Http\Controllers;

use Illuminate\Validation\ValidationException;
use Illuminate\Http\Request;
use App\Traits\Autenticable;
use App\Models\Cliente;
use App\Models\Pais;
use DB;
use Logger;

class ClientesController extends Controller
{
    use Autenticable;

    public function ingresar(Request $request)
    {
        $datos = $request->json()->all();

        $token = is_null($request->header('token')) ? $datos['token'] : $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $respuesta = [];

        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            $datos,
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        try {
            foreach ($datos['clientes'] as $cliente) {
                try {
                    $this->validateArray($request, $cliente, [
                        'nombre_rz' => 'required|max:100',
                        'nombre_fantasia' => 'required|max:100',
                        'rut' => 'required|rut|max:100',
                        'cliente' => 'required|integer|in:0,1',
                        'proveedor' => 'required|integer|in:0,1',
                        'prospecto' => 'required|integer|in:0,1',
                        'extranjero' => 'required|integer|in:0,1',
                        'giro' => 'required|max:100',
                        'direccion' => 'required|max:150',
                        'telefono' => 'required|max:100',
                        'fax' => 'sometimes|required|max:100',
                        'comuna' => 'required|max:100',
                        'ciudad' => 'required|max:100',
                        'pais' => 'required',
                        'email' => 'required|email|max:100',
                        'web' => 'sometimes|required|max:100',
                        'observacion' => 'required|max:100',
                        'nombre_cto' => 'required|max:250',
                        'email_cto' => 'required|email|max:250',
                        'telefono_cto' => 'required|max:200',
                        'celular_cto' => 'sometimes|required|max:200',
                        'cargo_cto' => 'required|max:200',
                        'cobranza_cto' => 'required|integer|in:0,1,2',
                        'vendedor' => 'required|rut',
                    ]);
                } catch (ValidationException $e) {
                    $respuesta[] = [$cliente['rut'] => (array) $e->getResponse()->getData()];
                    continue;
                }

                $cliente['rut'] = trim(strtoupper($cliente['rut']));

                $empresa = Cliente::obtenCliente($cliente['rut'], $usuario->rut_empresa);
                if (!is_null($empresa)) {
                    $respuesta[] = [$cliente['rut'] => ['mensaje' => 'Empresa ya registrada.']];
                    continue;
                }

                $pais = Pais::obtenPaisPorNombre($cliente['pais']);
                if (is_null($pais)) {
                    $respuesta[] = [$cliente['rut'] => ['mensaje' => 'País invalido.']];
                    continue;
                }

                //DB::beginTransaction();

                $resultado = DB::table('clienteproveedor')
                    ->insert([
                        'RUT_EMPRESA' => $usuario->rut_empresa,
                        'RUT_CLIENTEPROVEEDOR' => $cliente['rut'],
                        'RZ_CLIENTEPROVEEDOR' => $cliente['nombre_rz'],
                        'FANTASIA_CLIENTEPROVEEDOR' => $cliente['nombre_fantasia'],
                        'GIRO_CLIENTEPROVEEDOR' => $cliente['giro'],
                        'TELEFONO_CLIENTEPROVEEDOR' => $cliente['telefono'],
                        'FAX_CLIENTEPROVEEDOR' => isset($cliente['fax']) ? $cliente['fax'] : '',
                        'EMAIL_CLIENTEPROVEEDOR' => $cliente['email'],
                        'PROVEEDOR_CLIENTEPROVEEDOR' => $cliente['proveedor'] ? 'SI' : 'NO',
                        'CLIENTE_CLIENTEPROVEEDOR' => $cliente['cliente'] ? 'SI' : 'NO',
                        'PROSPECTO_CLIENTEPROVEEDOR' => $cliente['prospecto'] ? 'SI' : 'NO',
                        'OBSERVACION_CLIENTEPROVEEDOR' => $cliente['observacion'],
                        'ID_ESTADO' => 1,
                        'FECHACREACION' => date('Y-m-d'),
                        'FECHAUPDATE' => date('Y-m-d'),
                        'EXTRANJERO_CLIENTEPROVEEDOR' => $cliente['extranjero'] ? 'SI' : 'NO',
                        'ID_FPAGO' => 0,
                        'ID_VENDEDOR' => $cliente['vendedor'],
                        'CLA' => 0,
                        'PAGINA_WEB' => isset($cliente['web']) ? $cliente['web'] : '',
                    ]);

                if (!$resultado) {
                    $respuesta[] = [$cliente['rut'] => ['mensaje' => 'Cliente no pudo ser ingresado']];
                    DB::rollback();
                    continue;
                }

                $resultado = DB::table('dir_clienteproveedor')
                    ->insert([
                        'ID_DIRECCION' => null,
                        'RUT_EMPRESA' => $usuario->rut_empresa,
                        'RUT_CLIENTEPROVEEDOR' => $cliente['rut'],
                        'CORTO_DIRECCION' => 'CASA MATRIZ',
                        'DIRECCION_DIRECCION' => $cliente['direccion'],
                        'ID_COMUNA' => $cliente['comuna'],
                        'ID_CIUDAD' => $cliente['ciudad'],
                        'EMAIL_DIRECCION' => $cliente['email'],
                        'ID_PAIS' => $pais->codigo,
                    ]);

                if (!$resultado) {
                    $respuesta[] = [$cliente['rut'] => ['mensaje' => 'Direccion cliente no pudo ser ingresado']];
                    DB::rollback();
                    continue;
                }

                $resultado = DB::table('clienteproveedor_contacto')
                    ->insert([
                        'idcontacto' => null,
                        'fecha_creacion' => date('Y-m-d'),
                        'rut_empresa' => $usuario->rut_empresa,
                        'rut_cliente' => $cliente['rut'],
                        'nombre' => $cliente['nombre_cto'],
                        'cargo' => $cliente['cargo_cto'],
                        'email' => $cliente['email_cto'],
                        'telefono_oficina' => $cliente['telefono_cto'],
                        'telefono_celular' => isset($cliente['celular_cto']) ? $cliente['celular_cto'] : '',
                        'cobranza' => $cliente['cobranza_cto'],
                        'estado' => 1,
                    ]);

                if (!$resultado) {
                    $respuesta[] = [$cliente['rut'] => ['mensaje' => 'Contacto cliente no pudo ser ingresado']];
                    DB::rollback();
                    continue;
                }

                // Commit the transaction
                //DB::commit();

                $respuesta[] = [$cliente['rut'] => ['mensaje' => 'Cliente ingresado con exito.']];
            }
        } catch (\Exception $e) {
            //DB::rollback();
            //continue;
            //(array) $e->getResponse()->getData()
            $respuesta[] = ['Error' => ['mensaje' => 'Array de clientes vacio.']];

        }



        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            json_encode(utf8_encode_recursivo((array)$request->all())),
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        return response()->json($respuesta);
    }

    public function index(Request $request)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $empresas = DB::table('clienteproveedor')
            ->select(
                'clienteproveedor.RUT_CLIENTEPROVEEDOR AS rut',
                'clienteproveedor.RZ_CLIENTEPROVEEDOR AS nombre_rz',
                'clienteproveedor.FANTASIA_CLIENTEPROVEEDOR AS nombre_fantasia',
                'clienteproveedor.GIRO_CLIENTEPROVEEDOR AS giro',
                'clienteproveedor.TELEFONO_CLIENTEPROVEEDOR AS telefono',
                'clienteproveedor.EMAIL_CLIENTEPROVEEDOR AS email',
                'clienteproveedor.FAX_CLIENTEPROVEEDOR AS fax',
                'clienteproveedor.PROVEEDOR_CLIENTEPROVEEDOR AS proveedor',
                'clienteproveedor.CLIENTE_CLIENTEPROVEEDOR AS cliente',
                'clienteproveedor.PROSPECTO_CLIENTEPROVEEDOR AS prospecto',
                'clienteproveedor.OBSERVACION_CLIENTEPROVEEDOR AS observacion',
                'clienteproveedor.EXTRANJERO_CLIENTEPROVEEDOR AS extranjero',
                'clienteproveedor.ID_VENDEDOR AS vendedor',
                'clienteproveedor.PAGINA_WEB AS web'
            )
            ->whereRaw('clienteproveedor.RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
            ->get();

        return response()->json((array) $empresas);
    }

    public function clienteContactos(Request $request)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $empresas = DB::table('clienteproveedor')
            ->select(
                'clienteproveedor.RUT_CLIENTEPROVEEDOR AS rut',
                'clienteproveedor.RZ_CLIENTEPROVEEDOR AS nombre_rz',
                'clienteproveedor.FANTASIA_CLIENTEPROVEEDOR AS nombre_fantasia',
                'clienteproveedor.GIRO_CLIENTEPROVEEDOR AS giro',
                'clienteproveedor.TELEFONO_CLIENTEPROVEEDOR AS telefono',
                'clienteproveedor.EMAIL_CLIENTEPROVEEDOR AS email',
                'clienteproveedor.FAX_CLIENTEPROVEEDOR AS fax',
                'clienteproveedor.PROVEEDOR_CLIENTEPROVEEDOR AS proveedor',
                'clienteproveedor.CLIENTE_CLIENTEPROVEEDOR AS cliente',
                'clienteproveedor.PROSPECTO_CLIENTEPROVEEDOR AS prospecto',
                'clienteproveedor.OBSERVACION_CLIENTEPROVEEDOR AS observacion',
                'clienteproveedor.EXTRANJERO_CLIENTEPROVEEDOR AS extranjero',
                'clienteproveedor.ID_VENDEDOR AS vendedor',
                'clienteproveedor.PAGINA_WEB AS web'
            )
            ->whereRaw('clienteproveedor.RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
            ->get();


        foreach ($empresas as $key => $value) {

            $value->contactos = Cliente::obtenContactos($value->rut, $usuario->rut_empresa);
        }
        //dd($empresas);
        return response()->json((array) $empresas);
    }

    public function listarProveedores(Request $request)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $empresas = DB::table('clienteproveedor')
            ->select(
                'clienteproveedor.RUT_CLIENTEPROVEEDOR AS rut',
                'clienteproveedor.RZ_CLIENTEPROVEEDOR AS nombre_rz',
                'clienteproveedor.FANTASIA_CLIENTEPROVEEDOR AS nombre_fantasia',
                'clienteproveedor.GIRO_CLIENTEPROVEEDOR AS giro',
                'clienteproveedor.TELEFONO_CLIENTEPROVEEDOR AS telefono',
                'clienteproveedor.EMAIL_CLIENTEPROVEEDOR AS email',
                'clienteproveedor.FAX_CLIENTEPROVEEDOR AS fax',
                'clienteproveedor.PROVEEDOR_CLIENTEPROVEEDOR AS proveedor',
                'clienteproveedor.CLIENTE_CLIENTEPROVEEDOR AS cliente',
                'clienteproveedor.PROSPECTO_CLIENTEPROVEEDOR AS prospecto',
                'clienteproveedor.OBSERVACION_CLIENTEPROVEEDOR AS observacion',
                'clienteproveedor.EXTRANJERO_CLIENTEPROVEEDOR AS extranjero',
                'clienteproveedor.ID_VENDEDOR AS vendedor',
                'clienteproveedor.PAGINA_WEB AS web'
            )
            ->whereRaw('clienteproveedor.RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
            ->where('clienteproveedor.PROVEEDOR_CLIENTEPROVEEDOR', '!=', '')
            ->get();

        return response()->json((array) $empresas);
    }

    public function getClienteDireccion(Request $request)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $empresas = DB::table('clienteproveedor')
            ->select(
                'clienteproveedor.RUT_CLIENTEPROVEEDOR AS rut',
                'clienteproveedor.RZ_CLIENTEPROVEEDOR AS nombre_rz',
                'clienteproveedor.FANTASIA_CLIENTEPROVEEDOR AS nombre_fantasia',
                'clienteproveedor.GIRO_CLIENTEPROVEEDOR AS giro',
                'clienteproveedor.TELEFONO_CLIENTEPROVEEDOR AS telefono',
                'clienteproveedor.EMAIL_CLIENTEPROVEEDOR AS email',
                'clienteproveedor.FAX_CLIENTEPROVEEDOR AS fax',
                'clienteproveedor.PROVEEDOR_CLIENTEPROVEEDOR AS proveedor',
                'clienteproveedor.CLIENTE_CLIENTEPROVEEDOR AS cliente',
                'clienteproveedor.PROSPECTO_CLIENTEPROVEEDOR AS prospecto',
                'clienteproveedor.OBSERVACION_CLIENTEPROVEEDOR AS observacion',
                'clienteproveedor.EXTRANJERO_CLIENTEPROVEEDOR AS extranjero',
                'clienteproveedor.ID_VENDEDOR AS vendedor',
                'clienteproveedor.PAGINA_WEB AS web'
            )
            ->whereRaw('clienteproveedor.RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
            ->get();

        foreach ($empresas as $key => $cliente) {
            $direcciones = Cliente::obtenDirecciones($cliente->rut, $usuario->rut_empresa);

            $cliente->direccion = $direcciones;
        }

        return response()->json((array) $empresas);
    }

    public function ingresarDireccion(Request $request, $rut)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $respuesta = [];

        $empresa = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if (is_null($empresa)) {
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }

        foreach ($datos as $direccion) {
            try {
                $this->validateArray($request, $direccion, [
                    'direccion' => 'required|max:150',
                    'comuna' => 'required|max:100',
                    'ciudad' => 'required|max:100',
                    'pais' => 'required',
                    'email' => 'required|email|max:100',
                ]);
            } catch (ValidationException $e) {
                $respuesta[] = [$rut => (array) $e->getResponse()->getData()];
                continue;
            }

            $pais = Pais::obtenPaisPorNombre($direccion['pais']);
            if (is_null($pais)) {
                $respuesta[] = [$rut => ['mensaje' => 'País invalido.']];
                continue;
            }

            $resultado = DB::table('dir_clienteproveedor')
                ->insert([
                    'ID_DIRECCION' => null,
                    'RUT_EMPRESA' => $usuario->rut_empresa,
                    'RUT_CLIENTEPROVEEDOR' => $rut,
                    'CORTO_DIRECCION' => 'SUCURSAL',
                    'DIRECCION_DIRECCION' => $direccion['direccion'],
                    'ID_COMUNA' => $direccion['comuna'],
                    'ID_CIUDAD' => $direccion['ciudad'],
                    'EMAIL_DIRECCION' => $direccion['email'],
                    'ID_PAIS' => $pais->codigo,
                ]);

            if (!$resultado) {
                $respuesta[] = [$rut => ['mensaje' => 'Direccion cliente no pudo ser ingresado']];
                continue;
            }

            $respuesta[] = [$rut => ['mensaje' => 'Direccion ingresada con exito.']];
        }

        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            json_encode(utf8_encode_recursivo((array)$request->all())),
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        return response()->json($respuesta);
    }

    public function indexDireccion(Request $request, $rut)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $respuesta = [];

        $empresa = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if (is_null($empresa)) {
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }

        $direcciones = Cliente::obtenDirecciones($rut, $usuario->rut_empresa);

        return response()->json($direcciones);
    }

    public function actualizarDireccion(Request $request, $rut, $id)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        try {
            $this->validate($request, [
                'direccion' => 'sometimes|required|max:150',
                'comuna' => 'sometimes|required|max:100',
                'ciudad' => 'sometimes|required|max:100',
                'pais' => 'sometimes|required',
                'email' => 'sometimes|required|email|max:100',
            ]);
        } catch (ValidationException $e) {
            $respuesta[] = [$rut => (array) $e->getResponse()->getData()];
            //continue;
        }

        $direccion = Cliente::obtenDireccion($id, $rut, $usuario->rut_empresa);
        if(is_null($direccion)){
            $respuesta[] = [$rut => ['mensaje' => 'Direccion no registrada.']];
            return response()->json($respuesta);
        }

        $pais = isset($datos['pais']) ? $datos['pais'] : $direccion->pais;
        $pais = Pais::obtenPaisPorNombre($pais);
        if (is_null($pais)) {
            $respuesta[] = [$rut => ['mensaje' => 'País invalido.']];
            return response()->json($respuesta);
        }

        try{
            DB::table('dir_clienteproveedor')
                ->whereRaw('dir_clienteproveedor.RUT_CLIENTEPROVEEDOR = \''.$rut.'\'')
                ->whereRaw('dir_clienteproveedor.RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
                ->where('ID_DIRECCION', '=', $id)
                ->update([
                    'DIRECCION_DIRECCION' => isset($datos['direccion']) ? $datos['direccion'] : $direccion->direccion,
                    'ID_COMUNA' => isset($datos['comuna']) ? $datos['comuna'] : $direccion->comuna,
                    'ID_CIUDAD' => isset($datos['ciudad']) ? $datos['ciudad'] : $direccion->ciudad,
                    'EMAIL_DIRECCION' => isset($datos['email']) ? $datos['email'] : $direccion->email,
                    'ID_PAIS' => $pais->codigo,
                ]);
        }catch(Exception $e) {
            $respuesta[] = [$rut => "Error al actualizar direccion."];
            return response()->json($respuestas);
        }

        $respuesta[] = [$rut => ['mensaje' => 'Direccion actualizada con exito.']];

        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            json_encode(utf8_encode_recursivo((array)$request->all())),
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        return response()->json($respuesta);
    }

    public function actualizarClienteDireccion(Request $request, $rut)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        try {
            $this->validate($request, [
                'nombre_rz' => 'sometimes|required|max:100',
                'nombre_fantasia' => 'sometimes|required|max:100',
                'cliente' => 'sometimes|required|integer|in:0,1',
                'proveedor' => 'sometimes|required|integer|in:0,1',
                'prospecto' => 'sometimes|required|integer|in:0,1',
                'extranjero' => 'sometimes|required|integer|in:0,1',
                'giro' => 'sometimes|required|max:100',
                'telefono' => 'sometimes|required|max:100',
                'fax' => 'sometimes|sometimes|required|max:100',
                'web' => 'sometimes|required|url|max:100',
                'email' => 'sometimes|required|max:100',                
                'observacion' => 'sometimes|required|max:100',
                'vendedor' => 'sometimes|required|rut',
                'direccion' => 'sometimes|required|max:150',
                'comuna' => 'sometimes|required|max:100',
                'ciudad' => 'sometimes|required|max:100',
                'pais' => 'sometimes|required',
                'email' => 'sometimes|required|email|max:100',
            ]);
        } catch (ValidationException $e) {
            $respuesta[] = [$rut => (array) $e->getResponse()->getData()];
            return response()->json($respuesta);
        }

        $cliente = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if(is_null($cliente)){
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }
        
        try{
            DB::table('clienteproveedor')
                ->whereRaw('RUT_CLIENTEPROVEEDOR = \''.$rut.'\'')
                ->whereRaw('RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
                ->update([
                    'RZ_CLIENTEPROVEEDOR' => isset($datos['nombre_rz']) ? $datos['nombre_rz'] : $cliente->RZ_CLIENTEPROVEEDOR,
                    'FANTASIA_CLIENTEPROVEEDOR' => isset($datos['nombre_fantasia']) ? $datos['nombre_fantasia'] : $cliente->FANTASIA_CLIENTEPROVEEDOR,
                    'GIRO_CLIENTEPROVEEDOR' => isset($datos['giro']) ? $datos['giro'] : $cliente->GIRO_CLIENTEPROVEEDOR,
                    'TELEFONO_CLIENTEPROVEEDOR' => isset($datos['telefono']) ? $datos['telefono'] : $cliente->TELEFONO_CLIENTEPROVEEDOR,
                    'FAX_CLIENTEPROVEEDOR' => isset($datos['fax']) ? $datos['fax'] : $cliente->FAX_CLIENTEPROVEEDOR,
                    'PROVEEDOR_CLIENTEPROVEEDOR' => isset($datos['proveedor']) ? ($datos['proveedor'] == 1 ? 'SI' : 'NO') : $cliente->PROVEEDOR_CLIENTEPROVEEDOR,
                    'CLIENTE_CLIENTEPROVEEDOR' => isset($datos['cliente']) ? ($datos['cliente'] == 1 ? 'SI' : 'NO') : $cliente->CLIENTE_CLIENTEPROVEEDOR,
                    'PROSPECTO_CLIENTEPROVEEDOR' => isset($datos['prospecto']) ? ($datos['prospecto'] == 1 ? 'SI' : 'NO') : $cliente->PROSPECTO_CLIENTEPROVEEDOR,
                    'OBSERVACION_CLIENTEPROVEEDOR' => isset($datos['observacion']) ? $datos['observacion'] : $cliente->OBSERVACION_CLIENTEPROVEEDOR,
                    'EXTRANJERO_CLIENTEPROVEEDOR' => isset($datos['extranjero']) ? ($datos['extranjero'] == 1 ? 'SI' : 'NO') : $cliente->EXTRANJERO_CLIENTEPROVEEDOR,
                    'ID_VENDEDOR' => isset($datos['vendedor']) ? $datos['vendedor'] : $cliente->ID_VENDEDOR,
                    'PAGINA_WEB' => isset($datos['web']) ? $datos['web'] : $cliente->PAGINA_WEB,
                    'EMAIL_CLIENTEPROVEEDOR' => isset($datos['email']) ? $datos['email'] : $cliente->EMAIL_CLIENTEPROVEEDOR,
                ]);
        }catch(Exception $e) {
            $respuesta[] = [$rut => "Error al actualizar empresa."];
            return response()->json($respuestas);
        }
        
        $pais = isset($datos['pais']) ? $datos['pais'] : $cliente->ID_PAIS;
        $pais = DB::table('paises')
            ->where('descripcion', '=', $pais)
            ->first();

        if (is_null($pais)) {
            $respuesta[] = [$rut => ['mensaje' => 'País invalido.']];
            return response()->json($respuesta);
        }

        try{
            DB::table('dir_clienteproveedor')
                ->whereRaw('dir_clienteproveedor.RUT_CLIENTEPROVEEDOR = \''.$rut.'\'')
                ->whereRaw('dir_clienteproveedor.RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
                ->update([
                    'DIRECCION_DIRECCION' => isset($datos['direccion']) ? $datos['direccion'] : $cliente->direccion,
                    'ID_COMUNA' => isset($datos['comuna']) ? $datos['comuna'] : $cliente->comuna,
                    'ID_CIUDAD' => isset($datos['ciudad']) ? $datos['ciudad'] : $cliente->ciudad,
                    'EMAIL_DIRECCION' => isset($datos['email']) ? $datos['email'] : $cliente->email,
                    'ID_PAIS' => $pais->codigo,
                ]);
        }catch(Exception $e) {
            $respuesta[] = [$rut => "Error al actualizar direccion."];
            return response()->json($respuestas);
        }

        $respuesta[] = [$rut => ['mensaje' => 'Empresa actualizada con exito.']];

        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            json_encode(utf8_encode_recursivo((array)$request->all())),
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        return response()->json($respuesta);
    }

    public function actualizarCliente(Request $request, $rut)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        try {
            $this->validate($request, [
                'nombre_rz' => 'sometimes|required|max:100',
                'nombre_fantasia' => 'sometimes|required|max:100',
                'cliente' => 'sometimes|required|integer|in:0,1',
                'proveedor' => 'sometimes|required|integer|in:0,1',
                'prospecto' => 'sometimes|required|integer|in:0,1',
                'extranjero' => 'sometimes|required|integer|in:0,1',
                'giro' => 'sometimes|required|max:100',
                'telefono' => 'sometimes|required|max:100',
                'fax' => 'sometimes|sometimes|required|max:100',
                'web' => 'sometimes|required|url|max:100',
                'email' => 'sometimes|required|max:100',                
                'observacion' => 'sometimes|required|max:100',
                'vendedor' => 'sometimes|required|rut',
            ]);
        } catch (ValidationException $e) {
            $respuesta[] = [$rut => (array) $e->getResponse()->getData()];
            return response()->json($respuesta);
        }

        $cliente = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if(is_null($cliente)){
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }

        try{
            DB::table('clienteproveedor')
                ->whereRaw('RUT_CLIENTEPROVEEDOR = \''.$rut.'\'')
                ->whereRaw('RUT_EMPRESA = \''.$usuario->rut_empresa.'\'')
                ->update([
                    'RZ_CLIENTEPROVEEDOR' => isset($datos['nombre_rz']) ? $datos['nombre_rz'] : $cliente->RZ_CLIENTEPROVEEDOR,
                    'FANTASIA_CLIENTEPROVEEDOR' => isset($datos['nombre_fantasia']) ? $datos['nombre_fantasia'] : $cliente->FANTASIA_CLIENTEPROVEEDOR,
                    'GIRO_CLIENTEPROVEEDOR' => isset($datos['giro']) ? $datos['giro'] : $cliente->GIRO_CLIENTEPROVEEDOR,
                    'TELEFONO_CLIENTEPROVEEDOR' => isset($datos['telefono']) ? $datos['telefono'] : $cliente->TELEFONO_CLIENTEPROVEEDOR,
                    'FAX_CLIENTEPROVEEDOR' => isset($datos['fax']) ? $datos['fax'] : $cliente->FAX_CLIENTEPROVEEDOR,
                    'PROVEEDOR_CLIENTEPROVEEDOR' => isset($datos['proveedor']) ? ($datos['proveedor'] == 1 ? 'SI' : 'NO') : $cliente->PROVEEDOR_CLIENTEPROVEEDOR,
                    'CLIENTE_CLIENTEPROVEEDOR' => isset($datos['cliente']) ? ($datos['cliente'] == 1 ? 'SI' : 'NO') : $cliente->CLIENTE_CLIENTEPROVEEDOR,
                    'PROSPECTO_CLIENTEPROVEEDOR' => isset($datos['prospecto']) ? ($datos['prospecto'] == 1 ? 'SI' : 'NO') : $cliente->PROSPECTO_CLIENTEPROVEEDOR,
                    'OBSERVACION_CLIENTEPROVEEDOR' => isset($datos['observacion']) ? $datos['observacion'] : $cliente->OBSERVACION_CLIENTEPROVEEDOR,
                    'EXTRANJERO_CLIENTEPROVEEDOR' => isset($datos['extranjero']) ? ($datos['extranjero'] == 1 ? 'SI' : 'NO') : $cliente->EXTRANJERO_CLIENTEPROVEEDOR,
                    'ID_VENDEDOR' => isset($datos['vendedor']) ? $datos['vendedor'] : $cliente->ID_VENDEDOR,
                    'PAGINA_WEB' => isset($datos['web']) ? $datos['web'] : $cliente->PAGINA_WEB,
                    'EMAIL_CLIENTEPROVEEDOR' => isset($datos['email']) ? $datos['email'] : $cliente->EMAIL_CLIENTEPROVEEDOR,
                ]);
        }catch(Exception $e) {
            $respuesta[] = [$rut => "Error al actualizar empresa."];
            return response()->json($respuestas);
        }

        $respuesta[] = [$rut => ['mensaje' => 'Empresa actualizada con exito.']];

        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            json_encode(utf8_encode_recursivo((array)$request->all())),
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        return response()->json($respuesta);
    }

    public function obtenCliente(Request $request, $rut)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $respuesta = [];

        $empresa = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if (is_null($empresa)) {
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }

        //indicar direccion
        $direcciones = Cliente::obtenDirecciones($rut, $usuario->rut_empresa);

        //obtiene contactos
        $contactos = Cliente::obtenContactos($rut, $usuario->rut_empresa);
        return response()->json([
            'rut' => $empresa->RUT_CLIENTEPROVEEDOR,
            'nombre_rz' => $empresa->RZ_CLIENTEPROVEEDOR,
            'nombre_fantasia' => $empresa->FANTASIA_CLIENTEPROVEEDOR,
            'giro' => $empresa->GIRO_CLIENTEPROVEEDOR,
            'telefono' => $empresa->TELEFONO_CLIENTEPROVEEDOR,
            'web' => $empresa->PAGINA_WEB,
            'mail' => $empresa->EMAIL_CLIENTEPROVEEDOR,
            'fax' => $empresa->FAX_CLIENTEPROVEEDOR,
            'proveedor' => $empresa->PROVEEDOR_CLIENTEPROVEEDOR,
            'cliente' => $empresa->CLIENTE_CLIENTEPROVEEDOR,
            'prospecto' => $empresa->PROSPECTO_CLIENTEPROVEEDOR,
            'extranjero' => $empresa->EXTRANJERO_CLIENTEPROVEEDOR,
            'observacion' => $empresa->OBSERVACION_CLIENTEPROVEEDOR,
            'vendedor' => $empresa->ID_VENDEDOR,
            'direccion' => $direcciones,
            'contacto' => $contactos,
            'web' => $empresa->PAGINA_WEB,
        ]);
    }

    public function indexContacto(Request $request, $rut)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $respuesta = [];

        $empresa = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if (is_null($empresa)) {
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }

        $contactos = Cliente::obtenContactos($rut, $usuario->rut_empresa);

        return response()->json($contactos);
    }

    public function ingresarContacto(Request $request, $rut)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $respuesta = [];

        $empresa = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if (is_null($empresa)) {
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }

        foreach ($datos as $contacto) {
            try {
                $this->validateArray($request, $contacto, [
                    'nombre_cto' => 'required|max:250',
                    'email_cto' => 'required|email|max:250',
                    'telefono_cto' => 'required|max:200',
                    'celular_cto' => 'sometimes|required|max:200',
                    'cargo_cto' => 'required|max:200',
                    'cobranza_cto' => 'required|integer|in:0,1,2',
                ]);
            } catch (ValidationException $e) {
                $respuesta[] = [$rut => (array) $e->getResponse()->getData()];
                continue;
            }

            $resultado = DB::table('clienteproveedor_contacto')
                ->insert([
                    'idcontacto' => null,
                    'fecha_creacion' => date('Y-m-d'),
                    'rut_empresa' => $usuario->rut_empresa,
                    'rut_cliente' => $rut,
                    'nombre' => $contacto['nombre_cto'],
                    'cargo' => $contacto['cargo_cto'],
                    'email' => $contacto['email_cto'],
                    'telefono_oficina' => $contacto['telefono_cto'],
                    'telefono_celular' => isset($contacto['celular_cto']) ? $contacto['celular_cto'] : '',
                    'cobranza' => $contacto['cobranza_cto'],
                    'estado' => 1,
                ]);

            if (!$resultado) {
                $respuesta[] = [$rut => ['mensaje' => 'Contacto no pudo ser ingresado']];
                return response()->json($respuesta);
            }

            $respuesta[] = [$rut => ['mensaje' => 'Contacto ingresado con exito.']];
        }

        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            json_encode(utf8_encode_recursivo((array)$request->all())),
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        return response()->json($respuesta);
    }

    public function obtenContacto(Request $request, $rut, $id)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        $respuesta = [];

        $contacto = Cliente::obtenContacto($id, $rut, $usuario->rut_empresa);
        if (is_null($contacto)) {
            $respuesta[] = [$rut => ['mensaje' => 'Contacto no registrado.']];
            return response()->json($respuesta);
        }

        return response()->json((array) $contacto);
    }

    public function actualizarContacto(Request $request, $rut, $id)
    {
        $datos = $request->json()->all();

        $token = $request->header('token');

        $usuario = $this->valida_token($token);
        if (is_null($usuario)) {
            return $this->tokenInvalido();
        }

        try {
            $this->validate($request, [
                'nombre_cto' => 'sometimes|required|max:250',
                'email_cto' => 'sometimes|required|email|max:250',
                'telefono_cto' => 'sometimes|required|max:200',
                'celular_cto' => 'sometimes|sometimes|required|max:200',
                'cargo_cto' => 'sometimes|required|max:200',
                'cobranza_cto' => 'sometimes|required|integer|in:0,1,2',
            ]);
        } catch (ValidationException $e) {
            $respuesta[] = [$rut => (array) $e->getResponse()->getData()];
            //continue;
        }

        $cliente = Cliente::obtenCliente($rut, $usuario->rut_empresa);
        if(is_null($cliente)){
            $respuesta[] = [$rut => ['mensaje' => 'Empresa no registrada.']];
            return response()->json($respuesta);
        }

        $contacto = Cliente::obtenContacto($id, $rut, $usuario->rut_empresa);
        if (is_null($contacto)) {
            $respuesta[] = [$rut => ['mensaje' => 'Contacto no registrado.']];
            return response()->json($respuesta);
        }

        try{
            DB::table('clienteproveedor_contacto')
                ->whereRaw('rut_cliente = \''.$rut.'\'')
                ->whereRaw('rut_empresa = \''.$usuario->rut_empresa.'\'')
                ->where('idcontacto', '=', $id)
                ->update([
                    'nombre' => isset($datos['nombre_cto']) ? $datos['nombre_cto'] : $contacto->nombre_cto,
                    'cargo' => isset($datos['cargo_cto']) ? $datos['cargo_cto'] : $contacto->cargo_cto,
                    'email' => isset($datos['email_cto']) ? $datos['email_cto'] : $contacto->email_cto,
                    'telefono_oficina' => isset($datos['telefono_cto']) ? $datos['telefono_cto'] : $contacto->telefono_cto,
                    'telefono_celular' => isset($datos['celular_cto']) ? $datos['celular_cto'] : $contacto->celular_cto,
                    'cobranza' => isset($datos['cobranza_cto']) ? $datos['cobranza_cto'] : $contacto->cobranza_cto,
                ]);
        }catch(Exception $e) {
            $respuesta[] = [$rut => "Error al actualizar contacto."];
            return response()->json($respuestas);
        }

        $respuesta[] = [$rut => ['mensaje' => 'Contacto actualizado con exito.']];

        Logger::log(
            $usuario->rut_empresa,
            $request->path(),
            $request->method(),
            $usuario->rut_usuario,
            json_encode(utf8_encode_recursivo((array)$request->all())),
            $respuesta,
            ENV('AMBIENTE_SII'),
            $request->ip()
        );

        return response()->json($respuesta);
    }
}
